import invariant from 'tiny-invariant';
import { _TypedDataEncoder } from '@ethersproject/hash';
import { MaxSigDeadline, MaxOrderedNonce, MaxAllowanceTransferAmount, MaxAllowanceExpiration } from './constants';
import { permit2Domain } from './domain';
const PERMIT_DETAILS = [
    { name: 'token', type: 'address' },
    { name: 'amount', type: 'uint160' },
    { name: 'expiration', type: 'uint48' },
    { name: 'nonce', type: 'uint48' },
];
const PERMIT_TYPES = {
    PermitSingle: [
        { name: 'details', type: 'PermitDetails' },
        { name: 'spender', type: 'address' },
        { name: 'sigDeadline', type: 'uint256' },
    ],
    PermitDetails: PERMIT_DETAILS,
};
const PERMIT_BATCH_TYPES = {
    PermitBatch: [
        { name: 'details', type: 'PermitDetails[]' },
        { name: 'spender', type: 'address' },
        { name: 'sigDeadline', type: 'uint256' },
    ],
    PermitDetails: PERMIT_DETAILS,
};
function isPermit(permit) {
    return !Array.isArray(permit.details);
}
export class AllowanceTransfer {
    /**
     * Cannot be constructed.
     */
    constructor() { }
    // return the data to be sent in a eth_signTypedData RPC call
    // for signing the given permit data
    static getPermitData(permit, permit2Address, chainId) {
        invariant(MaxSigDeadline.gte(permit.sigDeadline), 'SIG_DEADLINE_OUT_OF_RANGE');
        const domain = permit2Domain(permit2Address, chainId);
        if (isPermit(permit)) {
            validatePermitDetails(permit.details);
            return {
                domain,
                types: PERMIT_TYPES,
                values: permit,
            };
        }
        else {
            permit.details.forEach(validatePermitDetails);
            return {
                domain,
                types: PERMIT_BATCH_TYPES,
                values: permit,
            };
        }
    }
    static hash(permit, permit2Address, chainId) {
        const { domain, types, values } = AllowanceTransfer.getPermitData(permit, permit2Address, chainId);
        return _TypedDataEncoder.hash(domain, types, values);
    }
}
function validatePermitDetails(details) {
    invariant(MaxOrderedNonce.gte(details.nonce), 'NONCE_OUT_OF_RANGE');
    invariant(MaxAllowanceTransferAmount.gte(details.amount), 'AMOUNT_OUT_OF_RANGE');
    invariant(MaxAllowanceExpiration.gte(details.expiration), 'EXPIRATION_OUT_OF_RANGE');
}
//# sourceMappingURL=allowanceTransfer.js.map